<?php

namespace App\Http\Controllers;

use App\Models\Exam;
use App\Models\Question;
use App\Models\StudentExam;
use App\Models\Payment;
use App\Models\Result;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;

class StudentController extends Controller
{
    public function dashboard()
    {
        $purchasedExams = StudentExam::where('user_id', Auth::id())->count();
        $passedExams = Result::whereHas('studentExam', function ($query) {
            $query->where('user_id', Auth::id());
        })->where('passed', true)->count();
        return view('student.dashboard', compact('purchasedExams', 'passedExams'));
    }

    public function exams()
    {
        $studentExams = StudentExam::with('exam', 'results')
        ->where('user_id', Auth::id())
        ->orderBy('created_at', 'desc') // Ensure the latest records come first
        ->get()
        ->unique(function ($item) {
            return $item->exam_id; // This will ensure that we only get the latest exam record per exam_id
        });

        // dd($studentExams);
        return view('student.exams', compact('studentExams'));
    }

    public function attemptExam($id)
    {
        // dd('hellow');
        $studentExam = StudentExam::where('user_id', Auth::id())->where('exam_id', $id)->latest()->firstOrFail();
        $exam = $studentExam->exam;
        if ($studentExam->attempts >= $exam->max_attempts && !$studentExam->results()->where('passed', true)->exists()) {
            return redirect()->route('exam.purchase', $id)->with('error', 'Max attempts reached. Please repurchase.');
        }

        $questionCount = $exam->total_mark / $exam->mark_per_question;
        $questions = Question::where('exam_id', $id)->inRandomOrder()->take($questionCount)->get();
        return view('student.exam-attempt', compact('exam', 'questions', 'studentExam'));
    }

    public function submitExam(Request $request, $id)
    {
        $studentExam = StudentExam::where('user_id', Auth::id())->where('exam_id', $id)->latest()->firstOrFail();
        $exam = $studentExam->exam;
        $answers = $request->input('answers', []);
        $score = 0;
        $questionCount = $exam->total_mark / $exam->mark_per_question;

        $questions = Question::where('exam_id', $id)->take($questionCount)->get();
        foreach ($questions as $question) {
            if (isset($answers[$question->id]) && $answers[$question->id] == $question->correct_option) {
                $score += $exam->mark_per_question;
            }
        }

        $passed = $score >= $exam->pass_mark;
        $studentExam->increment('attempts');

        Result::create([
            'student_exam_id' => $studentExam->id,
            'score' => $score,
            'passed' => $passed,
            'answers' => $answers,
        ]);

        return view('student.result', compact('exam', 'score', 'passed'));
    }

    public function payments()
    {
        $payments = Payment::where('user_id', Auth::id())->with('exam')->get();
        return view('student.payments', compact('payments'));
    }

    public function settings()
    {
        return view('student.settings');
    }

    public function updateSettings(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'password' => 'nullable|confirmed|min:6',
        ]);

        $user = Auth::user();
        $user->name = $request->name;
        $user->email = $request->email;
        if ($request->password) {
            $user->password = Hash::make($request->password);
        }
        $user->save();

        return redirect()->route('student.settings')->with('success', 'Settings updated successfully!');
    }


    public function viewResult($id)
{
    $studentExam = StudentExam::where('user_id', Auth::id())
        ->where('exam_id', $id)
        ->with(['exam', 'results' => function ($query) {
            $query->latest();
        }])
        ->latest()
        ->firstOrFail();

        // dd($studentExam);

    $latestResult = $studentExam->results()->latest()->first();

    if (!$latestResult) {
        return redirect()->route('student.exams')->with('error', 'No result found for this exam.');
    }

    return view('student.view-result', compact('studentExam', 'latestResult'));
}

}
