<?php

namespace App\Http\Controllers;

use App\Models\Coupon;
use App\Models\Exam;
use App\Models\Question;
use App\Models\User;
use App\Models\Payment;
use App\Models\SubTopic;
use App\Models\testcategory;
use App\Models\topic;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Facades\Validator;


class AdminController extends Controller
{
    public function showAdminLogin()
    {
        return view('auth.admin-login');
    }

    public function adminLogin(Request $request)
    {
        $credentials = $request->validate([
            'email' => 'required|email',
            'password' => 'required',
        ]);

        if (Auth::attempt($credentials) && Auth::user()->role === 'admin') {
            return redirect()->route('admin.dashboard');
        }

        return back()->withErrors(['email' => 'Invalid admin credentials']);
    }

    public function dashboard()
    {
        $totalStudents = User::where('role', 'student')->count();
        $totalExams = Exam::count();
        $passedStudents = \App\Models\Result::where('passed', true)->distinct('student_exam_id')->count();
        $revenue = Payment::sum('amount');
        return view('admin.dashboard', compact('totalStudents', 'totalExams', 'passedStudents', 'revenue'));
    }

    public function exams()
    {
        $exams = Exam::all();
        $topics = topic::all();
        $testCategories = testcategory::all();

        return view('admin.exams', compact('exams','topics','testCategories'));
    }







    public function topic()
    {
        $topics = topic::all();
        // dd($topics);
        return view('admin.topics', compact('topics'));
    }

    public function showCreateTopic()
    {
        return view('admin.Topic-create');
    }

    public function CreateTopic(Request $request)
    {
        $data = $request->validate([
            'topic_name' => 'required|string|max:255',
            'topic_image' => 'nullable|image|max:2048',
            'topic_Percentage' => 'required|integer|min:1',
        ]);

        if ($request->hasFile('topic_image')) {
            $data['topic_image'] = $request->file('topic_image')->store('Topics', 'public');
        }

        topic::create($data);
        return redirect()->route('admin.topic')->with('success', 'Topic created successfully!');
    }

    public function showEditTopic($id)
    {
        $topic = topic::findOrFail($id);
        return view('admin.topic-edit', compact('topic'));
    }

    public function updateTopic(Request $request, $id)
{

    $topic = topic::findOrFail($id);
    // dd($exam);

    if ($request->hasFile('topic_image')) {
        // Delete old image if exists
        if ($topic->topic_image && Storage::disk('public')->exists($topic->topic_image)) {
            Storage::disk('public')->delete($topic->topic_image);
        }

        // Store new image
        $path = $request->file('topic_image')->store('Topics', 'public');
        $topic->topic_image = $path;
    }

    // Update the exam
    $topic->topic_name = $request->topic_name;
    $topic->topic_Percentage = $request->topic_Percentage;
    $topic->save();

    return redirect()->route('admin.topic')->with('success', 'Topic updated successfully!');
}

public function subtopic($topicId)
{
    $topic = Topic::findOrFail($topicId);
    $subtopics = SubTopic::where('topic_id', $topicId)->get();

    return view('admin.subtopics-index', compact('topic', 'subtopics'));
}

public function createsubtopic($topicId)
    {
        $topic = Topic::findOrFail($topicId);
        return view('admin.subtopics-create', compact('topic'));
    }

    public function storesubtopic(Request $request, $topicId)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        SubTopic::create([
            'topic_id' => $topicId,
            'name' => $request->name,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.subtopics.index', $topicId)->with('success', 'Sub-topic created successfully.');
    }

    public function editsubtopic($id)
    {
        $subTopic = SubTopic::findOrFail($id);
        return view('admin.subtopics-edit', compact('subTopic'));
    }

    public function updatesubtopic(Request $request, $id)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'description' => 'nullable|string',
        ]);

        $subTopic = SubTopic::findOrFail($id);
        $subTopic->update([
            'name' => $request->name,
            'description' => $request->description,
        ]);

        return redirect()->route('admin.subtopics.index', $subTopic->topic_id)->with('success', 'Sub-topic updated successfully.');
    }










public function showCreateExam()
    {
        $topics = topic::all();
        $testCategories = testcategory::all();
        return view('admin.exam-create',compact('topics', 'testCategories'));
    }



    public function createExam(Request $request)
    {
    //    dd($request);

        $data = $request->validate([
            'name' => 'required|string|max:255',
            'amount' => 'required|numeric|min:0',
            'topic_id' => 'required',
            'test_category_id' => 'required',
            'image' => 'nullable|image|max:2048',
            'max_attempts' => 'required|integer|min:1',
            'total_mark' => 'required|integer|min:1',
            'mark_per_question' => 'required|integer|min:1',
            'pass_mark' => 'required|integer|min:1',
            'duration_minutes' => 'required|integer|min:1',
            'hard_ques' => 'required|integer'
        ]);

        if ($request->hasFile('image')) {
            $data['image'] = $request->file('image')->store('exams', 'public');
        }

        Exam::create($data);
        return redirect()->route('admin.exams')->with('success', 'Exam created successfully!');
    }

    public function showEditExam($id)
    {
        $exam = Exam::findOrFail($id);
        $topics = topic::all();
        $testCategories = testcategory::all();
        return view('admin.exam-edit', compact('exam','topics', 'testCategories'));
    }

    public function updateExam(Request $request, $id)
{
    $exam = Exam::findOrFail($id);

    // Check for image upload and replace old image
    if ($request->hasFile('image')) {
        if ($exam->image && Storage::disk('public')->exists($exam->image)) {
            Storage::disk('public')->delete($exam->image);
        }
        $path = $request->file('image')->store('exams', 'public');
        $exam->image = $path;
    }

    // Update basic exam details
    $exam->name = $request->name;
    $exam->amount = $request->amount;
    $exam->topic_id = $request->topic_id;
    $exam->test_category_id = $request->test_category_id;
    $exam->max_attempts = $request->max_attempts;
    $exam->total_mark = $request->total_mark;
    $exam->mark_per_question = $request->mark_per_question;
    $exam->duration_minutes = $request->duration_minutes;
    $exam->pass_mark = $request->pass_mark;
    $exam->hard_ques = $request->hard_ques;

    // Check if user wants to activate the exam
    if ($request->has('is_active')) {
        // Total required questions
        $requiredQuestions = intval($exam->total_mark / $exam->mark_per_question);
        $actualQuestions = Question::where('exam_id', $exam->id)->count();

        if ($actualQuestions < $requiredQuestions) {
            return redirect()->back()->with('error', "Not able to activate test. Minimum total questions not reached. Required: $requiredQuestions, Found: $actualQuestions.");
        }

        // Required hard questions
        $requiredHard = intval($exam->hard_ques);
        $actualHard = Question::where('exam_id', $exam->id)->where('is_hard', 1)->count();

        if ($actualHard < $requiredHard) {
            return redirect()->back()->with('error', "Not able to activate test. Minimum hard questions not reached. Required: $requiredHard, Found: $actualHard.");
        }

        // Required normal questions = total - hard
        $requiredNormal = $requiredQuestions - $requiredHard;
        $actualNormal = Question::where('exam_id', $exam->id)->where('is_hard', 0)->count();

        if ($actualNormal < $requiredNormal) {
            return redirect()->back()->with('error', "Not able to activate test. Minimum normal (non-hard) questions not reached. Required: $requiredNormal, Found: $actualNormal.");
        }

        $exam->is_active = 1;
    } else {
        $exam->is_active = 0;
    }


    $exam->save();

    return redirect()->route('admin.exams')->with('success', 'Exam updated successfully!');
}


    public function questions($id)
    {
        $exam = Exam::findOrFail($id);
        $questions = $exam->questions;
        return view('admin.questions', compact('exam', 'questions'));
    }

    public function showCreateQuestion($id)
    {
        $exam = Exam::findOrFail($id);
        return view('admin.question-create', compact('exam'));
    }

    public function createQuestion(Request $request, $id)
    {
    //    dd($request);

        $request->validate([
            'question' => 'required|string',
            'options' => 'required|array|min:4|max:4',
            'correct_option' => 'required|integer|min:0|max:3',

        ]);

        Question::create([
            'exam_id' => $id,
            'question' => $request->question,
            'options' => $request->options,
            'correct_option' => $request->correct_option,
            'is_hard' => $request->has('is_hard') ? 1 : 0,
        ]);

        return redirect()->route('admin.questions', $id)->with('success', 'Question added successfully!');
    }

    public function students()
    {
        $students = User::where('role', 'student')->get();
        return view('admin.students', compact('students'));
    }

    public function payments()
    {
        $payments = Payment::with(['user', 'exam'])->get();
        return view('admin.payments', compact('payments'));
    }

    public function settings()
    {
        return view('admin.settings');
    }

    public function updateSettings(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email',
            'password' => 'nullable|confirmed|min:6',
        ]);

        $user = Auth::user();
        $user->name = $request->name;
        $user->email = $request->email;
        if ($request->password) {
            $user->password = Hash::make($request->password);
        }
        $user->save();

        return redirect()->route('admin.settings')->with('success', 'Settings updated successfully!');
    }

    public function edit(Question $question)
{
    $exam = $question->exam;
    return view('admin.question-edit', compact('question', 'exam'));
}

// Handle update
public function update(Request $request, Question $question)
{
    $validator = Validator::make($request->all(), [
        'question' => 'required|string',
        'options' => 'required|array|min:4|max:4',
        'correct_option' => 'required|integer|min:0|max:3',
    ]);

    if ($validator->fails()) {
        return back()->withErrors($validator)->withInput();
    }

    $question->update([
        'question' => $request->question,
        'options' => $request->options,
        'correct_option' => $request->correct_option,
    ]);

    return redirect()->route('admin.questions', $question->exam_id)->with('success', 'Question updated successfully!');
}

// Delete question
public function destroy(Question $question)
{
    $examId = $question->exam_id;
    $question->delete();

    return redirect()->route('admin.questions', $examId)->with('success', 'Question deleted successfully!');
}

public function couponcreate()
{
    return view('admin.coupons-create');
}

public function couponindex()
{
    $coupons = Coupon::latest()->get();
    return view('admin.coupons-index', compact('coupons'));
}


public function couponstore(Request $request)
{
    $request->validate([
        'code' => 'required|string|unique:coupons,code',
        'type' => 'required|in:percent,amount',
        'value' => 'required|numeric|min:0',
        'usage_limit' => 'required|integer|min:1',
        'remarks' => 'nullable|string',
    ]);

    $coupon = Coupon::create([
        'code' => strtoupper($request->code),
        'type' => $request->type,
        'value' => $request->value,
        'usage_limit' => $request->usage_limit,
        'remarks' => $request->remarks,
    ]);

    return redirect()->back()->with('success', 'Coupon created successfully! Code: ' . $coupon->code);
}


public function testcategory()
    {
        $testcategories = testcategory::all();
        return view('admin.testcategory', compact('testcategories'));
    }


public function showCreateTestcategory()
    {
        return view('admin.Testcategory-create');
    }

public function CreateTestcategory(Request $request)
    {
        $data = $request->validate([
            'testcategory_name' => 'required|string|max:255',
            'gcc_std' => 'required|numeric|min:0',
            'gcc_can' => 'required|numeric|min:0',
            'us_std' => 'required|numeric|min:0',
            'us_can' => 'required|numeric|min:0',
            'ind_std' => 'required|numeric|min:0',
            'ind_can' => 'required|numeric|min:0',

        ]);

        testcategory::create($data);

        return redirect()->route('admin.testcategory')->with('success', 'Test Category created successfully!');
    }

    public function showEditTestcategory($id)
    {
        $testcategory = testcategory::findOrFail($id);
        return view('admin.testcategory-edit', compact('testcategory'));
    }

    public function updateTestcategory(Request $request, $id)
{

    $testcategory = testcategory::findOrFail($id);
    // dd($exam);

    // Update the exam
    $testcategory->testcategory_name = $request->testcategory_name;
    $testcategory->gcc_std = $request->gcc_std;
    $testcategory->gcc_can = $request->gcc_can;
    $testcategory->us_std = $request->us_std;
    $testcategory->us_can = $request->us_can;
    $testcategory->ind_std = $request->ind_std;
    $testcategory->ind_can = $request->ind_can;

    $testcategory->save();

    return redirect()->route('admin.testcategory')->with('success', 'Test Category updated successfully!');
}



}
